<?php
/**
 * DOM document class.
 *
 * @package WC_Shipping_USPS
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Drop in replacement for DOMDocument that is secure against XML eXternal Entity (XXE) Injection.
 * Bails if any DOCTYPE is found
 *
 * Comments in quotes come from the DOMDocument documentation: http://php.net/manual/en/class.domdocument.php
 */
class WC_Safe_DOMDocument extends DOMDocument {
	/**
	 * When called non-statically (as an object method) with malicious data, no Exception is thrown, but the object is emptied of all DOM nodes.
	 *
	 * @param string $filename The path to the XML document.
	 * @param int    $options Bitwise OR of the libxml option constants. http://us3.php.net/manual/en/libxml.constants.php.
	 *
	 * @return bool|DOMDocument true on success, false on failure.  If called statically (E_STRICT error), returns DOMDocument on success.
	 *
	 * @throws Exception When the file is empty or not readable.
	 */
	public function load( $filename, $options = 0 ) {
		if ( '' === $filename ) {
			// "If an empty string is passed as the filename or an empty file is named, a warning will be generated."
			// "This warning is not generated by libxml and cannot be handled using libxml's error handling functions."
			throw new Exception( 'WC_Safe_DOMDocument::load(): Empty string supplied as input' );
		}

		if ( ! is_file( $filename ) || ! is_readable( $filename ) ) {
			// This warning probably would have been generated by libxml and could have been handled handled using libxml's error handling functions.
			// In WC_Safe_DOMDocument, however, we catch it before libxml, so it can't.
			// The alternative is to let file_get_contents() handle the error, but that's annoying.
			throw new Exception( esc_html( 'WC_Safe_DOMDocument::load(): I/O warning : failed to load external entity "' . $filename . '"' ) );
		}

		if ( is_object( $this ) ) {
			// phpcs:ignore --- still need to use this function
			return $this->loadXML( file_get_contents( $filename ), $options );
		} else {
			// "This method *may* be called statically, but will issue an E_STRICT error."
			return self::loadXML( file_get_contents( $filename ), $options ); // phpcs:ignore --- still need to use this function
		}
	}

	/**
	 * When called non-statically (as an object method) with malicious data, no Exception is thrown, but the object is emptied of all DOM nodes.
	 *
	 * @param string $source The string containing the XML.
	 * @param int    $options Bitwise OR of the libxml option constants. http://us3.php.net/manual/en/libxml.constants.php.
	 *
	 * @return bool|DOMDocument true on success, false on failure.  If called statically (E_STRICT error), returns DOMDocument on success.
	 *
	 * @throws Exception When the file is empty or not readable.
	 */
	public function loadXML( $source, $options = 0 ) {
		if ( '' === $source ) {
			// "If an empty string is passed as the source, a warning will be generated."
			// "This warning is not generated by libxml and cannot be handled using libxml's error handling functions."
			throw new Exception( 'WC_Safe_DOMDocument::load(): Empty string supplied as input' );
		}

		$old = null;

		if ( function_exists( 'libxml_disable_entity_loader' ) && PHP_MAJOR_VERSION < 8 ) {
			$old = libxml_disable_entity_loader( true ); // phpcs:ignore --- only works for PHP version < 8
		}

		$return = parent::loadXML( $source, $options );

		if ( ! is_null( $old ) ) {
			libxml_disable_entity_loader( $old ); // phpcs:ignore --- only works for PHP version < 8
		}

		if ( ! $return ) {
			return $return;
		}

		// "This method *may* be called statically, but will issue an E_STRICT error.".
		$is_this = is_object( $this );

		$object = $is_this ? $this : $return;

		if ( isset( $object->doctype ) ) {
			if ( $is_this ) {
				// Get rid of the dangerous input by removing *all* nodes.
				while ( $this->firstChild ) {  // phpcs:ignore --- Need to use camelcase to follow DOM document class variable name
					$this->removeChild( $this->firstChild ); // phpcs:ignore --- Need to use camelcase to follow DOM document class variable name
				}
			}

			throw new Exception( 'WC_Safe_DOMDocument::loadXML(): Unsafe DOCTYPE Detected' );
		}

		return $return;
	}
}
